<?php

namespace App\Http\Controllers;

use App\Models\Student;
use App\Models\UniversityCost;
use App\Models\StudentCostAllocation;
use Illuminate\Http\Request;

class CostAllocationController extends Controller
{
    public function index()
    {
        $allocations = StudentCostAllocation::with(['student', 'cost'])
            ->paginate(25);
            
        return view('admin.cost-allocations.index', compact('allocations'));
    }

    public function create()
    {
        $students = Student::all();
        $costs = UniversityCost::all();
        return view('admin.cost-allocations.create', compact('students', 'costs'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'student_id' => 'required|exists:students,id',
            'cost_id' => 'required|exists:university_costs,id',
            'student_share' => 'required|numeric|min:0',
            'government_share' => 'required|numeric|min:0',
            'academic_year' => 'required|max:9|regex:/^\d{4}\/\d{4}$/',
        ]);

        // Validate total allocation
        $cost = UniversityCost::find($validated['cost_id']);
        $total = $validated['student_share'] + $validated['government_share'];
        
        if($total != $cost->amount) {
            return back()->withErrors([
                'allocation' => 'Student share + Government share must equal the total cost ('.$cost->amount.')'
            ]);
        }

        StudentCostAllocation::create($validated);

        return redirect()->route('admin.cost-allocations.index')
            ->with('success', 'Cost allocation created successfully');
    }

    public function edit(StudentCostAllocation $costAllocation)
    {
        $students = Student::all();
        $costs = UniversityCost::all();
        return view('admin.cost-allocations.edit', compact('costAllocation', 'students', 'costs'));
    }

    public function update(Request $request, StudentCostAllocation $costAllocation)
    {
        $validated = $request->validate([
            'student_share' => 'required|numeric|min:0',
            'government_share' => 'required|numeric|min:0',
            'academic_year' => 'required|max:9|regex:/^\d{4}\/\d{4}$/',
        ]);

        // Validate total allocation
        $total = $validated['student_share'] + $validated['government_share'];
        if($total != $costAllocation->cost->amount) {
            return back()->withErrors([
                'allocation' => 'Student share + Government share must equal the total cost ('.$costAllocation->cost->amount.')'
            ]);
        }

        $costAllocation->update($validated);

        return redirect()->route('admin.cost-allocations.index')
            ->with('success', 'Cost allocation updated successfully');
    }

    public function destroy(StudentCostAllocation $costAllocation)
    {
        $costAllocation->delete();
        return redirect()->route('admin.cost-allocations.index')
            ->with('success', 'Cost allocation deleted successfully');
    }
}