<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Country;
use App\Models\Region;
use App\Models\City;
use App\Models\Woreda;
use App\Models\Language;
use App\Models\Customer;
use App\Models\Policy;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class CustomerController extends Controller
{

    public function index()
    {
        // Option A: order by first_name and last_name separately (recommended)
        $customers = Customer::orderBy('first_name')
            ->orderBy('last_name')
            ->get();

        // Option B: order by full name concatenated (less performant, but single order clause)
        // $customers = Customer::orderByRaw("CONCAT(first_name, ' ', last_name) ASC")->get();

        return view('admin.customers.index', compact('customers'));
    }

    public function create()
    {
        $countries = Country::all();
        $languages = Language::where('is_active', 1)->get();

        return view('admin.customers.create', compact('countries', 'languages'));
    }

    public function getRegions(Request $request)
    {
        $regions = Region::where('country_id', $request->country_id)->get();
        return response()->json($regions);
    }

    public function getCities(Request $request)
    {
        $cities = City::where('region_id', $request->region_id)->get();
        return response()->json($cities);
    }

    public function getWeredas(Request $request)
    {
        $weredas = Woreda::where('city_id', $request->city_id)->get();
        return response()->json($weredas);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            // Customer validation
            'first_name' => 'required|string|max:100',
            'middle_name' => 'nullable|string|max:100',
            'last_name' => 'required|string|max:100',
            'gender' => 'required|in:male,female,other',
            'date_of_birth' => 'required|date',
            'place_of_birth' => 'required|string|max:255',
            'marital_status' => 'required|in:single,married,divorced,widowed',
            'nationality' => 'required|string|max:100',
            'national_id' => 'nullable|string|max:50',
            'passport_number' => 'nullable|string|max:50',
            'tax_id' => 'nullable|string|max:50',
            'email' => 'nullable|email|max:100|unique:customers,email',
            'phone' => 'required|string|max:20',
            'alt_phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'house_number' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'required|string|max:100',
            'administrative_region_city' => 'nullable|string|max:255',
            'sub_city_zone' => 'nullable|string|max:255',
            'wereda_town' => 'nullable|string|max:255',
            'kebele' => 'nullable|string|max:255',
            'postal_code' => 'nullable|string|max:20',
            'occupation' => 'nullable|string|max:100',
            'employer_name' => 'nullable|string|max:255',
            'employer_address' => 'nullable|string',
            'employment_status' => 'nullable|in:employed,unemployed,self-employed,retired,student',
            'annual_income' => 'nullable|numeric|min:0',
            'customer_status' => 'required|in:active,inactive,blacklisted',
            'notes' => 'nullable|string',
            'source' => 'required|in:walk-in,website,agent,referral',
            'password' => 'nullable|string|min:6',
            'preferred_contact_method' => 'nullable|in:email,phone,sms,whatsapp,telegram',
            'allow_marketing_emails' => 'nullable|boolean',
            'allow_sms_notifications' => 'nullable|boolean',
            'allow_push_notifications' => 'nullable|boolean',
            'language_preference' => 'nullable|string|max:50',
            'timezone' => 'nullable|string|max:100',

            // Policy validation
            'policy_number' => 'nullable|string|max:255|unique:policies,policy_number',
            'policy_type' => 'required|string|max:255',
            'coverage_type' => 'required|in:individual,family,group',
            'policy_name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'issue_date' => 'required|date',
            'maturity_date' => 'nullable|date|after_or_equal:start_date',
            'renewal_date' => 'nullable|date|after_or_equal:end_date',
            'sum_assured' => 'required|numeric|min:0',
            'premium_amount' => 'required|numeric|min:0',
            'premium_frequency' => 'required|in:monthly,quarterly,semi-annually,annually',
            'payment_method' => 'required|in:bank_transfer,credit_card,cash,mobile_money',
            'auto_renew' => 'nullable|boolean',
            'status' => 'required|in:active,inactive,cancelled,expired',
            'cancellation_reason' => 'nullable|string',
            'number_of_beneficiaries' => 'nullable|integer|min:0',
            'attachments' => 'nullable|string',
            'policy_notes' => 'nullable|string',
            'policy_source' => 'required|in:agent,branch,online,referral',
        ]);

        // Prepare password for customer, default if not provided
        $validated['password'] = bcrypt($validated['password'] ?? 'password');

        DB::beginTransaction();

        try {
            // Extract customer data only (exclude policy fields)
            $customerData = collect($validated)->only([
                'first_name',
                'middle_name',
                'last_name',
                'gender',
                'date_of_birth',
                'place_of_birth',
                'marital_status',
                'nationality',
                'national_id',
                'passport_number',
                'tax_id',
                'email',
                'phone',
                'alt_phone',
                'address',
                'house_number',
                'city',
                'state',
                'country',
                'administrative_region_city',
                'sub_city_zone',
                'wereda_town',
                'kebele',
                'postal_code',
                'occupation',
                'employer_name',
                'employer_address',
                'employment_status',
                'annual_income',
                'customer_status',
                'notes',
                'source',
                'password',
                'preferred_contact_method',
                'allow_marketing_emails',
                'allow_sms_notifications',
                'allow_push_notifications',
                'language_preference',
                'timezone'
            ])->toArray();

            $customerData['created_by'] = Auth::id();

            $customer = Customer::create($customerData);



            // Extract policy data only (exclude customer fields)
            $policyData = collect($validated)->only([
                'policy_number',
                'policy_type',
                'coverage_type',
                'policy_name',
                'description',
                'start_date',
                'end_date',
                'issue_date',
                'maturity_date',
                'renewal_date',
                'sum_assured',
                'premium_amount',
                'premium_frequency',
                'payment_method',
                'auto_renew',
                'status',
                'cancellation_reason',
                'number_of_beneficiaries',
                'attachments',
                'policy_notes',
                'policy_source'
            ])->toArray();

            $policyData['customer_id'] = $customer->id;
            $policyData['created_by'] = Auth::id();
            $policyData['updated_by'] = Auth::id();
            $policyData['policy_number'] = 'POL' . str_pad(mt_rand(1, 99999999), 8, '0', STR_PAD_LEFT);

            Policy::create($policyData);

            DB::commit();

            return response()->json(['message' => 'Customer and policy created successfully.']);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => 'Failed to create customer and policy.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function select2(Request $request)
    {
        $search = $request->input('term');

        $results = \App\Models\Customer::query()
            ->when($search, function ($query, $search) {
                $query->where('first_name', 'like', "%{$search}%")
                    ->orWhere('middle_name', 'like', "%{$search}%")
                    ->orWhere('last_name', 'like', "%{$search}%");
            })
            ->select('id', 'first_name', 'middle_name', 'last_name')
            ->limit(20)
            ->get()
            ->map(function ($customer) {
                return [
                    'id' => $customer->id,
                    'text' => trim("{$customer->first_name} {$customer->middle_name} {$customer->last_name}")
                ];
            });

        return response()->json(['results' => $results]);
    }
}
