<?php

namespace App\Http\Controllers;

use App\Models\StudentCostAllocation;
use App\Models\GovernmentContribution;
use Illuminate\Http\Request;

class GovernmentContributionController extends Controller
{
    public function index()
    {
        $contributions = GovernmentContribution::with('allocation.student')
            ->paginate(25);
            
        return view('admin.government-contributions.index', compact('contributions'));
    }

    public function create()
    {
        $allocations = StudentCostAllocation::where('government_share', '>', 0)
            ->doesntHave('governmentContributions')
            ->get();
            
        return view('admin.government-contributions.create', compact('allocations'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'allocation_id' => 'required|exists:student_cost_allocations,id',
            'amount' => 'required|numeric|min:0',
            'disbursement_date' => 'required|date',
        ]);

        // Verify allocation
        $allocation = StudentCostAllocation::find($validated['allocation_id']);
        if($validated['amount'] > $allocation->government_share) {
            return back()->withErrors([
                'amount' => 'Amount exceeds allocated government share ('.$allocation->government_share.')'
            ]);
        }

        GovernmentContribution::create([
            'allocation_id' => $validated['allocation_id'],
            'amount' => $validated['amount'],
            'disbursement_ref' => 'GOV-'.now()->format('YmdHis'),
            'disbursement_date' => $validated['disbursement_date'],
            'status' => 'approved',
        ]);

        return redirect()->route('admin.government-contributions.index')
            ->with('success', 'Government contribution recorded successfully');
    }

    public function show(GovernmentContribution $governmentContribution)
    {
        return view('admin.government-contributions.show', compact('governmentContribution'));
    }

    public function destroy(GovernmentContribution $governmentContribution)
    {
        $governmentContribution->delete();
        return redirect()->route('admin.government-contributions.index')
            ->with('success', 'Government contribution deleted successfully');
    }
}