<?php

namespace App\Http\Controllers;

use App\Models\StudentCostAllocation;
use App\Models\InstallmentPlan;
use Illuminate\Http\Request;
use Carbon\Carbon;

class InstallmentPlanController extends Controller
{
    public function index()
    {
        $plans = InstallmentPlan::with('allocation.student')
            ->paginate(25);
            
        return view('admin.installment-plans.index', compact('plans'));
    }

    public function create()
    {
        $allocations = StudentCostAllocation::whereDoesntHave('installments')
            ->get();
            
        return view('admin.installment-plans.create', compact('allocations'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'allocation_id' => 'required|exists:student_cost_allocations,id',
            'total_installments' => 'required|integer|min:2|max:12',
            'installment_amount' => 'required|numeric|min:1',
            'start_date' => 'required|date',
            'interval' => 'required|in:monthly,quarterly,semester',
        ]);

        // Validate total amount
        $allocation = StudentCostAllocation::find($validated['allocation_id']);
        $total = $validated['total_installments'] * $validated['installment_amount'];
        
        if($total != $allocation->student_share) {
            return back()->withErrors([
                'amount' => 'Total installments ('.$total.') must equal student share ('.$allocation->student_share.')'
            ]);
        }

        InstallmentPlan::create($validated);

        return redirect()->route('admin.installment-plans.index')
            ->with('success', 'Installment plan created successfully');
    }

    public function show(InstallmentPlan $installmentPlan)
    {
        return view('admin.installment-plans.show', compact('installmentPlan'));
    }

    public function generateSchedule(InstallmentPlan $installmentPlan)
    {
        $installments = [];
        $dueDate = Carbon::parse($installmentPlan->start_date);
        
        for($i = 1; $i <= $installmentPlan->total_installments; $i++) {
            $installments[] = [
                'due_date' => $dueDate->format('Y-m-d'),
                'amount' => $installmentPlan->installment_amount,
                'status' => 'pending',
            ];
            
            // Increment due date
            switch($installmentPlan->interval) {
                case 'monthly': $dueDate->addMonth(); break;
                case 'quarterly': $dueDate->addMonths(3); break;
                case 'semester': $dueDate->addMonths(6); break;
            }
        }
        
        $installmentPlan->installments()->createMany($installments);
        
        return back()->with('success', 'Installment schedule generated');
    }
}