<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use App\Models\Student;
use App\Models\StudentCostAllocation;
use App\Services\TelebirrService;
use App\Services\RevenueService;
use Illuminate\Http\Request;


class PaymentController extends Controller
{
    public function index()
    {
        $payments = Payment::with(['student', 'allocation'])
            ->latest()
            ->paginate(25);
            
        return view('admin.payments.index', compact('payments'));
    }

    public function create()
    {
        $students = Student::all();
        $allocations = StudentCostAllocation::all();
        return view('admin.payments.create', compact('students', 'allocations'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'student_id' => 'required|exists:students,id',
            'allocation_id' => 'required|exists:student_cost_allocations,id',
            'amount' => 'required|numeric|min:1',
        ]);

        // Create payment record
        $payment = Payment::create([
            'student_id' => $validated['student_id'],
            'allocation_id' => $validated['allocation_id'],
            'amount' => $validated['amount'],
            'status' => 'pending',
            'source' => 'student',
        ]);

        return redirect()->route('admin.payments.show', $payment)
            ->with('success', 'Payment created. Now process with Telebirr');
    }

    public function show(Payment $payment)
    {
        return view('admin.payments.show', compact('payment'));
    }

    public function telebirrTransactions()
    {
        $transactions = Payment::whereNotNull('telebirr_transaction_id')
            ->paginate(25);
            
        return view('admin.payments.telebirr', compact('transactions'));
    }

    public function initiateTelebirr(Payment $payment, TelebirrService $telebirr)
    {
        $student = $payment->student;
        
        $response = $telebirr->initiatePayment([
            'amount' => $payment->amount,
            'phone' => $student->phone,
            'reference' => 'PAY-'.$payment->id,
        ]);

        if(isset($response['payment_url'])) {
            return redirect($response['payment_url']);
        }

        return back()->withErrors(['telebirr' => 'Failed to initiate Telebirr payment']);
    }

    public function handleCallback(Request $request, TelebirrService $telebirr, RevenueService $revenue)
    {
        $transactionId = $request->input('transaction_id');
        $status = $telebirr->verifyPayment($transactionId);

        if($status === 'SUCCESS') {
            $payment = Payment::where('telebirr_reference', $request->ref)
                ->firstOrFail();
                
            $payment->update([
                'telebirr_transaction_id' => $transactionId,
                'status' => 'completed',
                'paid_at' => now(),
            ]);

            // Report to revenue system
            $reported = $revenue->reportPayment([
                'student_id' => $payment->student_id,
                'amount' => $payment->amount,
                'telebirr_id' => $transactionId,
            ]);

            if($reported) {
               // $payment->update(['revenue_receipt_id' => $revenue->getReceiptId()]);
            }
        }

        return view('payments.callback', compact('status'));
    }

    public function reminders()
    {
        // Payment reminder logic
    }
}