<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use App\Models\Student;
use App\Models\StudentCostAllocation;
use App\Models\GovernmentContribution;
use Illuminate\Http\Request;
use Carbon\Carbon;

class ReportController extends Controller
{
    public function financial(Request $request)
    {
        $start = $request->input('start', now()->subMonth()->format('Y-m-d'));
        $end = $request->input('end', now()->format('Y-m-d'));
        
        $studentPayments = Payment::where('source', 'student')
            ->whereBetween('paid_at', [$start, $end])
            ->sum('amount');
            
        $governmentFunding = GovernmentContribution::where('status', 'approved')
            ->whereBetween('disbursement_date', [$start, $end])
            ->sum('amount');
            
        return view('admin.reports.financial', compact('studentPayments', 'governmentFunding', 'start', 'end'));
    }

    public function studentBalances()
    {
        $students = Student::withSum('allocations', 'student_share')
            ->withSum('payments', 'amount')
            ->get()
            ->map(function($student) {
                $student->balance = $student->allocations_sum_student_share - $student->payments_sum_amount;
                return $student;
            });
            
        return view('admin.reports.student-balances', compact('students'));
    }

    public function govFunding()
    {
        $funding = GovernmentContribution::with('allocation.student')
            ->where('status', 'approved')
            ->orderBy('disbursement_date', 'desc')
            ->paginate(25);
            
        return view('admin.reports.gov-funding', compact('funding'));
    }

    public function tax()
    {
        $taxData = Payment::whereNotNull('tax_details')
            ->selectRaw('YEAR(paid_at) as year, MONTH(paid_at) as month, SUM(JSON_EXTRACT(tax_details, "$.amount")) as tax_amount')
            ->groupBy('year', 'month')
            ->orderBy('year', 'desc')
            ->orderBy('month', 'desc')
            ->paginate(12);
            
        return view('admin.reports.tax', compact('taxData'));
    }

    public function paymentChannels()
    {
        $channels = Payment::select('source')
            ->selectRaw('COUNT(*) as count, SUM(amount) as total')
            ->groupBy('source')
            ->get();
            
        return view('admin.reports.payment-channels', compact('channels'));
    }
}