<?php

namespace App\Http\Controllers;

use App\Models\Student;
use App\Models\UniversityProfile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class StudentController extends Controller
{
    public function index()
    {
        $students = Student::with('profile')->paginate(25);
        return view('admin.students.index', compact('students'));
    }

    public function create()
    {
        return view('admin.students.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'student_id' => 'required|unique:students|max:20',
            'full_name' => 'required|max:100',
            'email' => 'required|email|unique:students',
            'phone' => 'required|unique:students|max:15',
            'password' => 'required|min:8',
            'faculty' => 'required|max:100',
            'department' => 'required|max:100',
            'program' => 'required|max:100',
            'enrollment_year' => 'required|digits:4',
            'expected_graduation' => 'required|digits:4',
            'study_level' => 'required|in:undergraduate,masters,phd',
        ]);

        // Create student
        $student = Student::create([
            'student_id' => $validated['student_id'],
            'full_name' => $validated['full_name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
            'password' => Hash::make($validated['password']),
        ]);

        // Create profile
        $student->profile()->create([
            'faculty' => $validated['faculty'],
            'department' => $validated['department'],
            'program' => $validated['program'],
            'enrollment_year' => $validated['enrollment_year'],
            'expected_graduation' => $validated['expected_graduation'],
            'study_level' => $validated['study_level'],
        ]);

        return redirect()->route('admin.students.index')
            ->with('success', 'Student registered successfully');
    }

    public function show(Student $student)
    {
        return view('admin.students.show', compact('student'));
    }

    public function edit(Student $student)
    {
        return view('admin.students.edit', compact('student'));
    }

    public function update(Request $request, Student $student)
    {
        $validated = $request->validate([
            'full_name' => 'required|max:100',
            'email' => 'required|email|unique:students,email,'.$student->id,
            'phone' => 'required|max:15|unique:students,phone,'.$student->id,
            'faculty' => 'required|max:100',
            'department' => 'required|max:100',
            'program' => 'required|max:100',
            'study_level' => 'required|in:undergraduate,masters,phd',
            'is_tax_exempt' => 'boolean',
        ]);

        $student->update($validated);
        $student->profile()->update([
            'faculty' => $validated['faculty'],
            'department' => $validated['department'],
            'program' => $validated['program'],
            'study_level' => $validated['study_level'],
        ]);

        return redirect()->route('admin.students.index')
            ->with('success', 'Student updated successfully');
    }

    public function destroy(Student $student)
    {
        $student->delete();
        return redirect()->route('admin.students.index')
            ->with('success', 'Student deleted successfully');
    }

    // Additional methods
    public function profiles()
    {
        $profiles = UniversityProfile::with('student')->paginate(25);
        return view('admin.students.profiles', compact('profiles'));
    }

    public function import()
    {
        return view('admin.students.import');
    }

    public function processImport(Request $request)
    {
        // CSV import logic
    }

    public function groups()
    {
        // Student grouping logic
    }
}