<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Claim;
use App\Models\Customer;
use App\Models\Policy;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;


class ClaimController extends Controller
{
    // 🟢 Add this method to load the form view
    public function create()
    {
        $customers = Customer::select('id', 'first_name', 'middle_name', 'last_name')->get();
        return view('admin.claims.create', compact('customers'));
    }

    // 🟢 Handles AJAX form submission
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_id'            => 'required|exists:customers,id',
            'policy_id'              => 'required|exists:policies,id',
            'claim_type'             => 'required|string|max:100',
            'incident_date'          => 'required|date',
            'incident_location'      => 'nullable|string|max:255',
            'incident_description'   => 'required|string',
            'claim_amount_requested' => 'required|numeric|min:0',
            'adjuster_name'          => 'nullable|string|max:255',
            'adjuster_report'        => 'nullable|string',
            'documents.*'            => 'nullable|file|max:20480|mimes:pdf,jpg,jpeg,png,doc,docx',
        ]);
        

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed',
                'errors'  => $validator->errors()
            ], 422);
        }

        try {
            // Create claim
            $claim = Claim::create([
                'customer_id'             => $request->customer_id,
                'policy_id'               => $request->policy_id,
                'claim_type'              => $request->claim_type,
                'incident_date'           => $request->incident_date,
                'incident_location'       => $request->incident_location,
                'incident_description'    => $request->incident_description,
                'claim_amount_requested'  => $request->claim_amount_requested,
                'claim_status'            => 'pending', // Matches ENUM
                'assigned_adjuster'       => $request->adjuster_name,
                'adjuster_notes'          => $request->adjuster_report,
                'submitted_at'            => now(),
                'created_by'              => auth()->id(),
            ]);
            

            // Upload documents if any
            if ($request->hasFile('documents')) {
                foreach ($request->file('documents') as $file) {
                    $path = $file->store('claim_documents/' . $claim->id, 'public');

                    $claim->documents()->create([
                        'file_path'   => $path,
                        'file_name'   => $file->getClientOriginalName(),
                        'file_type'   => $file->getClientMimeType(),
                        'uploaded_by'=> auth()->id() ?? null,
                    ]);
                }
            }

            return response()->json([
                'message' => 'Claim submitted successfully!',
                'claim_id' => $claim->id,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Something went wrong',
                'error'   => $e->getMessage(),
                'trace'   => $e->getTrace()
            ], 500);
        }
    }

    public function getPolicy($id)
{
    $policy = Policy::with(['customer'])->findOrFail($id);
    
    return response()->json([
        'id' => $policy->id,
        'policy_number' => $policy->policy_number,
        'policy_type' => $policy->policy_type,
        'coverage_type' => $policy->coverage_type,
        'start_date' => $policy->start_date,
        'end_date' => $policy->end_date,
        'sum_insured' => $policy->sum_insured,
        'premium_amount' => $policy->premium_amount,
        'status' => $policy->status,
        'customer' => $policy->customer,
    ]);
}
}
