<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Payment;

use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\DataTables;
use App\Models\Student;
//use Carbon\CarbonPeriod;





class FinancialSummaryController extends Controller
{
    public function summary()
    {
        // Get today's total
        $totalSalesDaily = Payment::whereDate('created_at', today())->sum('amount');

        // Yesterday's total
        $previousDaily = Payment::whereDate('created_at', today()->subDay())->sum('amount');

        // This week's total (Mon to Sun)
        $totalSalesWeekly = Payment::whereBetween('created_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()])->sum('amount');

        // Last week's total
        $previousWeekly = Payment::whereBetween('created_at', [Carbon::now()->subWeek()->startOfWeek(), Carbon::now()->subWeek()->endOfWeek()])->sum('amount');

        // This month's total
        $totalSalesMonthly = Payment::whereYear('created_at', now()->year)
            ->whereMonth('created_at', now()->month)
            ->sum('amount');

        // Last month's total
        $previousMonthly = Payment::whereYear('created_at', now()->subMonth()->year)
            ->whereMonth('created_at', now()->subMonth()->month)
            ->sum('amount');

        // This year's total
        $totalSalesYearly = Payment::whereYear('created_at', now()->year)->sum('amount');

        // Last year's total
        $previousYearly = Payment::whereYear('created_at', now()->subYear()->year)->sum('amount');

        // Percent change calculations (prevent divide-by-zero)
        $dailyPercent = $previousDaily > 0 ? (($totalSalesDaily - $previousDaily) / $previousDaily) * 100 : 0;
        $weeklyPercent = $previousWeekly > 0 ? (($totalSalesWeekly - $previousWeekly) / $previousWeekly) * 100 : 0;
        $monthlyPercent = $previousMonthly > 0 ? (($totalSalesMonthly - $previousMonthly) / $previousMonthly) * 100 : 0;
        $yearlyPercent = $previousYearly > 0 ? (($totalSalesYearly - $previousYearly) / $previousYearly) * 100 : 0;

        return view('admin.financial.summary', compact(
            'totalSalesDaily',
            'totalSalesWeekly',
            'totalSalesMonthly',
            'totalSalesYearly',
            'dailyPercent',
            'weeklyPercent',
            'monthlyPercent',
            'yearlyPercent'
        ));
    }


    public function dailySummary(Request $request)
    {
        $date = $request->input('date', 'today');
        $targetDate = $date === 'yesterday' ? Carbon::yesterday() : Carbon::today();

        $start = $targetDate->copy()->startOfDay();
        $end = $targetDate->copy()->endOfDay();

        // 1. Hourly Chart Data (paid_at based)
        $payments = DB::table('payments')
            ->selectRaw('HOUR(paid_at) as hour, SUM(amount) as total')
            ->whereBetween('paid_at', [$start, $end])
            ->groupBy(DB::raw('HOUR(paid_at)'))
            ->orderBy('hour')
            ->pluck('total', 'hour')
            ->toArray();

        $hourlyTotals = [];
        for ($i = 0; $i < 24; $i++) {
            $hourlyTotals[] = isset($payments[$i]) ? round($payments[$i], 2) : 0;
        }

        // 2. Daily Summary Data (paid_at based)
        $total = DB::table('payments')
            ->whereDate('paid_at', $targetDate)
            ->sum('amount');

        $comparisonDate = $targetDate->copy()->subDay();

        $previous = DB::table('payments')
            ->whereDate('paid_at', $comparisonDate)
            ->sum('amount');

        $percent = $previous > 0 ? (($total - $previous) / $previous) * 100 : 0;

        return response()->json([
            'total' => round($total, 2),
            'totalFormatted' => number_format($total, 2),
            'percent' => number_format($percent, 2),
            'isUp' => $percent >= 0,
            'comparedTo' => $comparisonDate->toFormattedDateString(),
            'chart' => $hourlyTotals,
        ]);
    }




    public function weeklySummary(Request $request)
    {
        $range = $request->input('range', 'this'); // 'this' or 'last'

        $startOfWeek = $range === 'last' ? Carbon::now()->subWeek()->startOfWeek() : Carbon::now()->startOfWeek();
        $endOfWeek = $range === 'last' ? Carbon::now()->subWeek()->endOfWeek() : Carbon::now()->endOfWeek();

        // 1. Weekly Total
        $total = DB::table('payments')
            ->whereBetween('paid_at', [$startOfWeek, $endOfWeek])
            ->sum('amount');

        // 2. Previous Week Total for Comparison
        $comparisonStart = $startOfWeek->copy()->subWeek();
        $comparisonEnd = $endOfWeek->copy()->subWeek();

        $previous = DB::table('payments')
            ->whereBetween('paid_at', [$comparisonStart, $comparisonEnd])
            ->sum('amount');

        $percent = $previous > 0 ? (($total - $previous) / $previous) * 100 : 0;

        // 3. Chart Data: Sum by Day of Week (0 = Sunday, ..., 6 = Saturday)
        $daily = DB::table('payments')
            ->selectRaw('DAYOFWEEK(paid_at) as day, SUM(amount) as total')
            ->whereBetween('paid_at', [$startOfWeek, $endOfWeek])
            ->groupBy(DB::raw('DAYOFWEEK(paid_at)'))
            ->pluck('total', 'day')
            ->toArray();

        // Normalize to [0, 0, ..., 0] for Sunday–Saturday
        $dailyTotals = [];
        for ($i = 1; $i <= 7; $i++) {
            $dailyTotals[] = isset($daily[$i]) ? round($daily[$i], 2) : 0;
        }

        return response()->json([
            'total' => round($total, 2),
            'totalFormatted' => number_format($total, 2),
            'percent' => number_format($percent, 2),
            'isUp' => $percent >= 0,
            'chart' => $dailyTotals
        ]);
    }


    public function monthlySummary(Request $request)
    {
        $range = $request->input('range', 'this'); // 'this' or 'last'

        $startOfMonth = $range === 'last'
            ? Carbon::now()->subMonth()->startOfMonth()
            : Carbon::now()->startOfMonth();

        $endOfMonth = $range === 'last'
            ? Carbon::now()->subMonth()->endOfMonth()
            : Carbon::now()->endOfMonth();

        // Total for the month
        $total = DB::table('payments')
            ->whereBetween('paid_at', [$startOfMonth, $endOfMonth])
            ->sum('amount');

        // Previous month comparison
        $comparisonStart = $startOfMonth->copy()->subMonth();
        $comparisonEnd = $endOfMonth->copy()->subMonth();

        $previous = DB::table('payments')
            ->whereBetween('paid_at', [$comparisonStart, $comparisonEnd])
            ->sum('amount');

        $percent = $previous > 0 ? (($total - $previous) / $previous) * 100 : 0;

        // Group by each day
        $dailyData = DB::table('payments')
            ->selectRaw('DATE(paid_at) as date, SUM(amount) as total')
            ->whereBetween('paid_at', [$startOfMonth, $endOfMonth])
            ->groupBy('date')
            ->orderBy('date')
            ->pluck('total', 'date')
            ->toArray();

        // Fill in missing days with 0
        $labels = [];
        $data = [];

        $currentDay = $startOfMonth->copy();
        while ($currentDay->lte($endOfMonth)) {
            $date = $currentDay->format('Y-m-d');
            $labels[] = $currentDay->format('M d'); // ex: Jun 01
            $data[] = $dailyData[$date] ?? 0;
            $currentDay->addDay();
        }

        return response()->json([
            'total' => $total,
            'totalFormatted' => number_format($total, 2),
            'percent' => number_format($percent, 2),
            'isUp' => $percent >= 0,
            'comparedTo' => $range === 'last' ? now()->format('F Y') : now()->subMonth()->format('F Y'),
            'chart' => $data,
            'labels' => $labels,
        ]);
    }



    public function yearlySummary(Request $request)
    {
        $range = $request->input('range', 'this'); // 'this' or 'last'

        $startOfYear = $range === 'last'
            ? Carbon::now()->subYear()->startOfYear()
            : Carbon::now()->startOfYear();

        $endOfYear = $range === 'last'
            ? Carbon::now()->subYear()->endOfYear()
            : Carbon::now()->endOfYear();

        // Current year total sum
        $total = DB::table('payments')
            ->whereBetween('paid_at', [$startOfYear, $endOfYear])
            ->sum('amount');

        // Previous year range for comparison
        $comparisonStart = $startOfYear->copy()->subYear();
        $comparisonEnd = $endOfYear->copy()->subYear();

        $previous = DB::table('payments')
            ->whereBetween('paid_at', [$comparisonStart, $comparisonEnd])
            ->sum('amount');

        // Percentage change calculation
        $percent = $previous > 0 ? (($total - $previous) / $previous) * 100 : 0;

        // Monthly breakdown for current year
        $monthlyBreakdown = DB::table('payments')
            ->selectRaw("MONTH(paid_at) as month, SUM(amount) as total")
            ->whereBetween('paid_at', [$startOfYear, $endOfYear])
            ->groupByRaw("MONTH(paid_at)")
            ->pluck('total', 'month');

        // Ensure 12 months data
        $monthlyData = [];
        for ($i = 1; $i <= 12; $i++) {
            $monthlyData[] = round($monthlyBreakdown[$i] ?? 0, 2);
        }

        // Compared year: if 'this' year, compare to last year; if 'last' year, compare to two years ago
        if ($range === 'last') {
            $comparedTo = now()->subYear(2)->format('Y'); // e.g., 2023 if last year is 2024
        } else {
            $comparedTo = now()->subYear()->format('Y'); // e.g., 2023 if this year is 2024
        }

        return response()->json([
            'total' => round($total, 2),                 // raw number, for animation
            'percent' => round($percent, 2),             // raw number, no sign prefix here
            'isUp' => $percent >= 0,                      // boolean for up/down arrow
            'comparedTo' => $comparedTo,                  // string year label for comparison
            'monthlyData' => $monthlyData,                // 12 months data for chart
        ]);
    }


    public function yearlyMonthlyBreakdown(Request $request)
    {
        $range = $request->input('range', 'this'); // 'this' or 'last'

        $year = $range === 'last' ? now()->subYear()->year : now()->year;

        $monthlyTotals = DB::table('payments')
            ->selectRaw('MONTH(paid_at) as month, SUM(amount) as total')
            ->whereYear('paid_at', $year)
            ->groupByRaw('MONTH(paid_at)')
            ->orderBy('month')
            ->pluck('total', 'month');

        $monthlyData = [];
        for ($m = 1; $m <= 12; $m++) {
            $monthlyData[] = round($monthlyTotals[$m] ?? 0, 2);
        }

        return response()->json([
            'year' => $year,
            'monthlyTotals' => $monthlyData,
        ]);
    }

    public function recentTransactions(Request $request)
    {
        $payments = Payment::with(['student:id,student_id,full_name']) // student_id needed for join
            ->orderByDesc('paid_at');

        return DataTables::of($payments)
            ->addColumn('student_name', fn($p) => $p->student->full_name ?? 'Unknown')

            ->addColumn('status', fn($p) => 'success') // replace with logic if needed
            ->rawColumns(['student', 'status', 'checkbox'])
            ->make(true);
    }


    public function getPaymentCount(Request $request)
    {
        $period = $request->get('period', 'today');

        $query = Payment::query()->whereNotNull('paid_at'); // make sure only completed ones

        if ($period === 'today') {
            $query->whereDate('paid_at', now());
        } elseif ($period === 'week') {
            $query->whereBetween('paid_at', [now()->startOfWeek(), now()->endOfWeek()]);
        } elseif ($period === 'month') {
            $query->whereMonth('paid_at', now()->month)
                ->whereYear('paid_at', now()->year);
        }

        $count = $query->count();

        return response()->json([
            'count' => $count,
            'label' => ucfirst($period) . ' Transactions'
        ]);
    }

}
