<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Faker\Factory as Faker;
use Carbon\Carbon;

class StudentsTableSeeder extends Seeder
{
    public function run()
    {
        $faker = Faker::create();

        // Disable foreign key checks for safe truncation
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        DB::table('students')->truncate();
        DB::statement('SET FOREIGN_KEY_CHECKS=1;');

        // Get all universities from database
        $universities = DB::table('universities')->get(['id', 'code']);

        // Ethiopian names dataset
        $ethiopianNames = [
            'male' => [
                'Abebe', 'Alemu', 'Bekele', 'Dawit', 'Elias', 'Fikru', 'Girma', 'Habtamu',
                'Kebede', 'Lemma', 'Mekonnen', 'Nega', 'Solomon', 'Tadesse', 'Yohannes'
            ],
            'female' => [
                'Alemitu', 'Birtukan', 'Chaltu', 'Desta', 'Etenesh', 'Frehiwot', 'Genet',
                'Hirut', 'Kebebush', 'Lemlem', 'Mestawet', 'Netsanet', 'Selamawit', 'Tigist', 'Zeritu'
            ],
            'last' => [
                'Alemayehu', 'Assefa', 'Beyene', 'Demissie', 'Gebremichael', 'Hailu', 'Kassa',
                'Mengistu', 'Nigatu', 'Tesfaye', 'Worku', 'Yimer', 'Zewdie'
            ]
        ];

        // Track generated identifiers to prevent duplicates
        $usedTinNumbers = [];
        $usedStudentIds = [];
        $usedTelebirrIds = [];
        $usedPhones = [];
        $usedEmails = [];

        $records = [];
        $startYear = 2018; // For student IDs
        $endYear = 2025;   // For student IDs

        for ($i = 0; $i < 500; $i++) {
            $gender = $faker->randomElement(['Male', 'Female']);
            $firstName = $gender === 'Male'
                ? $faker->randomElement($ethiopianNames['male'])
                : $faker->randomElement($ethiopianNames['female']);

            $lastName = $faker->randomElement($ethiopianNames['last']);
            $fullName = $firstName . ' ' . $lastName;

            // Select random university
            $university = $faker->randomElement($universities);
            $universityId = $university->id;
            $universityCode = $university->code;

            // Generate unique TIN number (11 digits)
            do {
                $tinNumber = $faker->unique()->numerify('###########');
            } while (in_array($tinNumber, $usedTinNumbers));
            $usedTinNumbers[] = $tinNumber;

            // Generate unique student ID
            do {
                $studentId = $universityCode . '-' .
                             $faker->numberBetween($startYear, $endYear) .
                             '-' . $faker->unique()->numerify('#####');
            } while (in_array($studentId, $usedStudentIds));
            $usedStudentIds[] = $studentId;

            // Generate unique telebirr ID
            do {
                $telebirrId = 'TID' . $faker->regexify('[A-Z0-9]{20}');
            } while (in_array($telebirrId, $usedTelebirrIds));
            $usedTelebirrIds[] = $telebirrId;

            // Generate unique Ethiopian phone number
            do {
                $phone = '+2519' . $faker->numberBetween(10, 89) . $faker->numberBetween(100000, 999999);
            } while (in_array($phone, $usedPhones));
            $usedPhones[] = $phone;

            // Generate unique email
            do {
                $email = strtolower($firstName[0] . $lastName) . $faker->numerify('##') .
                         '@' . $faker->randomElement(['gmail.com', 'yahoo.com', 'student.edu.et']);
            } while (in_array($email, $usedEmails));
            $usedEmails[] = $email;

            $records[] = [
                'telebirr_open_id' => $telebirrId,
                'tin_number'       => $tinNumber,
                'student_id'       => $studentId,
                'university_id'    => $universityId,
                'full_name'        => $fullName,
                'gender'           => $gender,
                'birth_date'       => $faker->dateTimeBetween('-25 years', '-18 years')->format('Y-m-d'),
                'phone'            => $phone,
                'email'            => $email,
                'created_at'       => Carbon::now(),
                'updated_at'       => Carbon::now(),
            ];
        }

        // Insert in batches
        foreach (array_chunk($records, 50) as $chunk) {
            DB::table('students')->insert($chunk);
        }
    }
}