<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Faker\Factory as Faker;

class EthiopianCustomerSeeder extends Seeder
{
    public function run()
    {
        $faker = Faker::create();

        // Ethiopian name components
        $firstNames = [
            'Male' => ['Dawit', 'Ephrem', 'Girma', 'Kebede', 'Mulugeta', 'Nahom', 'Tewodros', 'Yohannes'],
            'Female' => ['Alemitu', 'Birtukan', 'Elsabet', 'Frehiwot', 'Genet', 'Hirut', 'Konjit', 'Meskel']
        ];

        $middleNames = ['Wolde', 'Gebre', 'Hailu', 'Tesfaye', 'Kassa', 'Demeke', 'Assefa', 'Birhanu'];
        
        $lastNames = [
            'Teshome', 'Gebremichael', 'Wolde', 'Hailu', 'Tadesse', 'Mekonnen', 
            'Assefa', 'Getachew', 'Abate', 'Zewdie', 'Mohammed', 'Ahmed'
        ];

        $usedCombinations = [];
        
        // Ethiopian data patterns
        $ethiopianCities = ['Addis Ababa', 'Dire Dawa', 'Bahir Dar', 'Mekelle', 'Hawassa'];
        $ethiopianRegions = ['Amhara', 'Oromia', 'Tigray', 'SNNPR', 'Somali'];

        for ($i = 0; $i < 100; $i++) {
            do {
                $gender = $faker->randomElement(['male', 'female']);
                $firstName = $faker->randomElement($firstNames[ucfirst($gender)]);
                $middleName = $faker->optional(0.7)->randomElement($middleNames); // 70% have middle names
                $lastName = $faker->randomElement($lastNames);
                $nameCombination = $firstName.$middleName.$lastName;
            } while (isset($usedCombinations[$nameCombination]));

            $usedCombinations[$nameCombination] = true;

            DB::table('customers')->insert([
                'first_name' => $firstName,
                'middle_name' => $middleName,
                'last_name' => $lastName,
                'gender' => $gender,
                'date_of_birth' => $this->generateEthiopianBirthdate(),
                'place_of_birth' => $faker->randomElement($ethiopianCities),
                'marital_status' => $faker->randomElement(['single', 'married', 'married', 'divorced']),
                'nationality' => 'ET',
                'national_id' => 'ETI'.$this->padUniqueNumber($i, 9),
                'passport_number' => 'ETP'.$this->padUniqueNumber($i, 8),
                'tax_id' => 'TIN'.$this->padUniqueNumber($i, 9),
                'email' => strtolower($firstName).'.'.$this->padUniqueNumber($i, 3).'@'.$faker->randomElement(['gmail.com', 'yahoo.com', 'ethionet.et']),
                'phone' => '+2519'.$this->padUniqueNumber($i, 8),
                'alt_phone' => $faker->optional(0.3)->numerify('+2511########'),
                'address' => 'Kebele '.$faker->numberBetween(1, 30).', '.$faker->streetName,
                'city' => $faker->randomElement($ethiopianCities),
                'state' => $faker->randomElement($ethiopianRegions),
                'country' => 'Ethiopia',
                'postal_code' => $faker->numerify('####'),
                'occupation' => $faker->randomElement(['Farmer', 'Teacher', 'Merchant', 'Government Worker']),
                'annual_income' => $faker->numberBetween(20000, 800000),
                'employment_status' => $faker->randomElement(['employed', 'self-employed', 'unemployed']),
                'customer_status' => $faker->randomElement(['active', 'active', 'inactive']),
                'source' => $faker->randomElement(['agent', 'walk-in', 'referral']),
                'password' => Hash::make('password'),
                'created_by' => 1,
                'updated_by' => 1,
                'preferred_contact_method' => $faker->randomElement(['phone', 'sms', 'whatsapp']),
                'language_preference' => $faker->randomElement(['am', 'om', 'en']),
                'timezone' => 'Africa/Addis_Ababa',
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
    }

    private function generateEthiopianBirthdate()
    {
        // 85% born after 1974 (Ethiopian calendar 1966)
        return fake()->dateTimeBetween(
            fake()->boolean(85) ? '-50 years' : '-80 years',
            '-18 years'
        )->format('Y-m-d');
    }

    private function padUniqueNumber($index, $length)
    {
        return str_pad($index + 1, $length, '0', STR_PAD_LEFT);
    }
}