<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Claim;
use App\Models\Customer;
use App\Models\Policy;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use Yajra\DataTables\Facades\DataTables;



class ClaimController extends Controller
{


        public function index()
    {

        return view('admin.claims.index');  // Only load the view here

    }

public function getClaimsData()
{
    $claims = Claim::with(['customer', 'policy'])->select('claims.*');

    return DataTables::eloquent($claims)
        ->addColumn('customer_name', fn($claim) => $claim->customer?->full_name ?? '—')
        ->addColumn('policy_number', fn($claim) => $claim->policy?->policy_number ?? '—')

        ->addColumn('status_pill', function ($claim) {
            $statusClasses = [
                'pending' => 'badge bg-warning text-dark',
                'approved' => 'badge bg-success',
                'rejected' => 'badge bg-danger',
                'under_review' => 'badge bg-info text-dark',
            ];
            $status = strtolower($claim->claim_status ?? 'pending');
            $class = $statusClasses[$status] ?? 'badge bg-secondary';

            return '<span class="' . $class . '">' . ucfirst($status) . '</span>';
        })

        ->addColumn('progress', function ($claim) {
            $progress = $claim->status_progress ?? 0;
            return '
                <div class="progress" style="height: 18px;">
                    <div class="progress-bar" role="progressbar" style="width: ' . $progress . '%;" aria-valuenow="' . $progress . '" aria-valuemin="0" aria-valuemax="100">
                        ' . $progress . '%
                    </div>
                </div>
            ';
        })

->editColumn('claim_amount_requested', function ($claim) {
    // Format as money with commas and two decimals
    $formatted = number_format($claim->claim_amount_requested, 2);
    // Add a <br> before the amount and use "Br" for Ethiopian Birr
    return '<br>Br ' . $formatted;
})


        ->addColumn('action', function ($claim) {
            $editUrl = route('admin.claims.edit', $claim->id);
            $deleteUrl = route('admin.claims.destroy', $claim->id);
             $approveUrl = route('admin.claims.destroy', $claim->id);

            return '
   <a href="#" class="text-success" title="Approve" data-bs-toggle="modal" data-bs-target="#approveModal" data-id="' . $claim->id . '">
                   <i class="icon-check"></i>
                </a>

                <a href="' . $editUrl . '" class="text-warning" title="Edit">
         <i class="icon-pencil-alt"></i>





                                           <a href="' . $deleteUrl . '" class="text-danger" title="Delete">
                   <i class="icon-trash"></i>
                </a>

            ';
        })

        ->rawColumns(['status_pill', 'progress', 'action', 'claim_amount_requested'])
        ->toJson();
}



    // 🟢 Add this method to load the form view
    public function create()
    {
        $customers = Customer::select('id', 'first_name', 'middle_name', 'last_name')->get();
        return view('admin.claims.create', compact('customers'));
    }

    // 🟢 Handles AJAX form submission
   public function store(Request $request)
{
    $request->validate([
        'customer_id' => 'required|exists:customers,id',
        'policy_id' => 'required|exists:policies,id',
        'claim_type' => 'required|string|max:255',
        'incident_date' => 'required|date',
        'incident_location' => 'required|string|max:255',
        'incident_description' => 'required|string',
        'claim_amount_requested' => 'required|numeric|min:0',
        'assigned_adjuster' => 'nullable|string|max:255',
        'adjuster_notes' => 'nullable|string',
    ]);

    // 🔑 Fetch the related policy
    $policy = Policy::findOrFail($request->policy_id);
    $lastClaimId = Claim::max('id') ?? 0;
    $claimNumber = 'CLM-' . str_pad($lastClaimId + 1, 8, '0', STR_PAD_LEFT);

    // 📝 Create the claim
Claim::create([
    'customer_id' => $request->customer_id,
    'corporate_customer_id' => $request->corporate_customer_id,
    'policy_id' => $request->policy_id,
    'policy_number' => $policy->policy_number,
    'claim_number' => $claimNumber, // ✅ Added here
    'claim_type' => $request->claim_type,
    'incident_date' => $request->incident_date,
    'incident_location' => $request->incident_location,
    'incident_description' => $request->incident_description,
    'claim_amount_requested' => $request->claim_amount_requested,
    'claim_status' => 'pending',
    'assigned_adjuster' => $request->assigned_adjuster,
    'adjuster_notes' => $request->adjuster_notes,
    'submitted_at' => now(),
    'created_by' => auth()->id(),
]);

    return redirect()->route('admin.claims.create')->with('success', 'Claim submitted successfully.');
}

    public function getPolicy($id)
{
    $policy = Policy::with(['customer'])->findOrFail($id);

    return response()->json([
        'id' => $policy->id,
        'policy_number' => $policy->policy_number,
        'policy_type' => $policy->policy_type,
        'coverage_type' => $policy->coverage_type,
        'start_date' => $policy->start_date,
        'end_date' => $policy->end_date,
        'sum_insured' => $policy->sum_insured,
        'premium_amount' => $policy->premium_amount,
        'status' => $policy->status,
        'customer' => $policy->customer,
    ]);
}

public function approve(Request $request, Claim $claim)
{
    // Your validation if needed
    // $request->validate([...]);

    // Mark claim as approved
    $claim->claim_status = 'approved';
    $claim->save();

    return response()->json(['message' => 'Claim approved successfully']);
}

}
