<?php

namespace App\Http\Controllers\Admin;




use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Models\Customer;
use App\Models\Lead;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Response;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Policy;
use DataTables;
use Carbon\Carbon;
use Carbon\CarbonPeriod;

use App\Models\Notification;
use Symfony\Component\HttpFoundation\StreamedResponse;

class DashboardController extends Controller
{
    public function index ()
    {
        return view('admin.dashboard.index');
    }


    public function claimsByStatus(Request $request)
    {
        $period = $request->get('period', 'monthly');
        $start = $request->get('start');
        $end = $request->get('end');

        $query = DB::table('claims');

        if ($start && $end) {
            $query->whereBetween('incident_date', [$start, $end]);
        } elseif ($period === 'weekly') {
            $query->whereBetween('incident_date', [now()->startOfWeek(), now()->endOfWeek()]);
        } elseif ($period === 'yearly') {
            $query->whereYear('incident_date', now()->year);
        } else {
            $query->whereMonth('incident_date', now()->month);
        }

        $rawCounts = $query
            ->select('claim_status', DB::raw('COUNT(*) as total'))
            ->groupBy('claim_status')
            ->pluck('total', 'claim_status')
            ->toArray();

        // Add all 6 statuses
        $statuses = ['pending', 'under_review', 'approved', 'rejected', 'paid', 'closed'];
        $labels = ['Pending', 'Under Review', 'Approved', 'Rejected', 'Paid', 'Closed'];
        $series = array_map(fn($s) => intval($rawCounts[$s] ?? 0), $statuses);

        return response()->json([
            'labels' => $labels,
            'series' => $series,
            'total' => array_sum($series),
        ]);
    }
    public function claimsByStatusList(Request $request)
    {
        $status = $request->get('status');
        $page = $request->get('page', 1);
        $perPage = 10;

        $claims = DB::table('claims')
            ->where('claim_status', $status)
            ->orderByDesc('submitted_at')
            ->paginate($perPage, ['*'], 'page', $page);

        return response()->json([
            'data' => $claims->items(),
            'pagination' => [
                'total' => $claims->total(),
                'per_page' => $claims->perPage(),
                'current_page' => $claims->currentPage(),
                'last_page' => $claims->lastPage(),
            ]
        ]);

    }



    public function getCustomerSummary(Request $request)
    {
        $status = $request->input('status');

        $query = Customer::query();
        if ($status && $status !== 'all') {
            $query->where('customer_status', $status);
        }

        $total = $query->count();

        $growth = $query->selectRaw('DATE_FORMAT(created_at, "%b %Y") as month, COUNT(*) as total')
            ->groupBy('month')
            ->orderByRaw('MIN(created_at)')
            ->limit(6)
            ->get();

        return response()->json([
            'total' => $total,
            'chart' => $growth,
            'status' => $status
        ]);
    }

    public function exportCustomers(Request $request)
    {
        $status = $request->input('status');
        $query = Customer::query();

        if ($status && $status !== 'all') {
            $query->where('customer_status', $status);
        }

        $customers = $query->get([
            'first_name', 'last_name', 'email', 'phone', 'customer_status', 'created_at'
        ]);

        $filename = 'customers_' . ($status ?? 'all') . '_' . now()->format('Ymd_His') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        $callback = function () use ($customers) {
            $handle = fopen('php://output', 'w');
            fputcsv($handle, ['First Name', 'Last Name', 'Email', 'Phone', 'Status', 'Registered At']);
            foreach ($customers as $customer) {
                fputcsv($handle, [
                    $customer->first_name,
                    $customer->last_name,
                    $customer->email,
                    $customer->phone,
                    ucfirst($customer->customer_status),
                    $customer->created_at->format('Y-m-d'),
                ]);
            }
            fclose($handle);
        };

        return response()->stream($callback, 200, $headers);
    }


public function policySummary(Request $request)
{
    $status = $request->status;

    $query = Policy::query();
    if ($status && $status !== 'all') {
        $query->where('status', $status);
    }

    $total = $query->count();

    // Group by Month (example: last 6 months)
    $chart = $query->selectRaw("DATE_FORMAT(created_at, '%b') as month, COUNT(*) as total")
                   ->where('created_at', '>=', now()->subMonths(6))
                   ->groupByRaw("month")
                   ->orderByRaw("MIN(created_at)")
                   ->get();

    return response()->json([
        'total' => $total,
        'status' => $status,
        'chart' => $chart,
    ]);
}

public function exportPolicies(Request $request)
{
    $status = $request->status;

    $query = Policy::query();
    if ($status && $status !== 'all') {
        $query->where('status', $status);
    }

    $policies = $query->get();

    // Export logic (can use Laravel Excel or CSV response)
    $filename = "policies_export_" . now()->format('Ymd_His') . ".csv";
    $headers = [
        "Content-type" => "text/csv",
        "Content-Disposition" => "attachment; filename=$filename",
        "Pragma" => "no-cache",
        "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
        "Expires" => "0"
    ];

    $callback = function () use ($policies) {
        $handle = fopen('php://output', 'w');
        fputcsv($handle, ['Policy Number', 'Status', 'Start Date', 'End Date', 'Premium']);

        foreach ($policies as $policy) {
            fputcsv($handle, [
                $policy->policy_number,
                $policy->status,
                $policy->start_date,
                $policy->end_date,
                $policy->premium_amount
            ]);
        }

        fclose($handle);
    };

    return response()->stream($callback, 200, $headers);
}



public function policyDrilldown(Request $request)
{
    $status = $request->input('status');
    $search = $request->input('search');

    $query = Policy::select('id', 'policy_number', 'policy_name', 'status', 'start_date', 'end_date', 'premium_amount')
        ->when($status && $status !== 'all', fn($q) => $q->where('status', $status))
        ->when($search, function ($q) use ($search) {
            $q->where(function ($query) use ($search) {
                $query->where('policy_number', 'like', "%{$search}%")
                      ->orWhere('policy_name', 'like', "%{$search}%");
            });
        });

    return $query->orderBy('created_at', 'desc')->paginate(10);

}

public function showDetails($id)
{
    $policy = Policy::with(['customer', 'claims'])->findOrFail($id);

    return response()->json([
        'customer_view' => view('admin.policies.tabs.customer', compact('policy'))->render(),
        'policy_view'   => view('admin.policies.tabs.policy', compact('policy'))->render(),
        'claims_view'   => view('admin.policies.tabs.claims', compact('policy'))->render(),
    ]);
}

public function premiumSummary(Request $request)
{
    $range = $request->input('range', 'all');
    $from = $request->input('from');
    $to = $request->input('to');

    $query = Policy::whereNull('deleted_at')->where('status', 'active');

    $label = 'Since Jan 2025';

    switch ($range) {
        case 'daily':
            $query->whereDate('created_at', now());
            $label = 'Today';
            break;
        case 'weekly':
            $query->whereBetween('created_at', [now()->startOfWeek(), now()]);
            $label = 'This Week';
            break;
        case 'monthly':
            $query->whereMonth('created_at', now()->month);
            $label = 'This Month';
            break;
        case 'yearly':
            $query->whereYear('created_at', now()->year);
            $label = 'This Year';
            break;
        case 'custom':
            if ($from && $to) {
                $query->whereBetween('created_at', [$from, $to]);
                $label = "From $from to $to";
            }
            break;
        default:
            break;
    }

    $total = $query->sum('premium_amount');

    // Fake chart data (replace with real grouping)
    $chart = collect(range(1, 30))->map(fn () => rand(20000, 50000));

    return response()->json([
        'total' => round($total),
        'chart' => $chart,
        'label' => $label,
    ]);
}

public function getStats(Request $request)
    {
        $range = $request->input('range', 'yearly');
        $from = $request->input('from');
        $to = $request->input('to');

        $now = now();

        // Define current and previous range
        switch ($range) {
            case 'daily':
                $currentFrom = $now->copy()->startOfDay();
                $currentTo = $now->copy()->endOfDay();
                $previousFrom = $currentFrom->copy()->subDay();
                $previousTo = $currentTo->copy()->subDay();
                break;

            case 'weekly':
                $currentFrom = $now->copy()->startOfWeek();
                $currentTo = $now->copy()->endOfWeek();
                $previousFrom = $currentFrom->copy()->subWeek();
                $previousTo = $currentTo->copy()->subWeek();
                break;

            case 'monthly':
                $currentFrom = $now->copy()->startOfMonth();
                $currentTo = $now->copy()->endOfMonth();
                $previousFrom = $currentFrom->copy()->subMonth();
                $previousTo = $currentTo->copy()->subMonth();
                break;

            case 'yearly':
            default:
                $currentFrom = $now->copy()->startOfYear();
                $currentTo = $now->copy()->endOfYear();
                $previousFrom = $currentFrom->copy()->subYear();
                $previousTo = $currentTo->copy()->subYear();
                break;
        }

        // If custom
        if ($range === 'custom' && $from && $to) {
            $currentFrom = Carbon::parse($from)->startOfDay();
            $currentTo = Carbon::parse($to)->endOfDay();
            $previousFrom = $currentFrom->copy()->subDays($currentFrom->diffInDays($currentTo));
            $previousTo = $currentFrom->copy()->subDay();
        }

        // Total Premium
        $currentPremium = Policy::whereBetween('created_at', [$currentFrom, $currentTo])->sum('premium_amount');
        $previousPremium = Policy::whereBetween('created_at', [$previousFrom, $previousTo])->sum('premium_amount');
        $premiumChange = $previousPremium == 0 ? 0 : round((($currentPremium - $previousPremium) / $previousPremium) * 100, 2);

        // New Leads
        $currentLeads = Lead::whereBetween('created_at', [$currentFrom, $currentTo])->count();
        $previousLeads = Lead::whereBetween('created_at', [$previousFrom, $previousTo])->count();
        $leadsChange = $previousLeads == 0 ? 0 : round((($currentLeads - $previousLeads) / $previousLeads) * 100, 2);

        // Policies Sold
        $currentPolicies = Policy::whereBetween('created_at', [$currentFrom, $currentTo])->count();
        $previousPolicies = Policy::whereBetween('created_at', [$previousFrom, $previousTo])->count();
        $policiesChange = $previousPolicies == 0 ? 0 : round((($currentPolicies - $previousPolicies) / $previousPolicies) * 100, 2);

        // Chart Data (Customers & Leads over time)
        $labels = [];
        $customerSeries = [];
        $leadSeries = [];

        $period = \Carbon\CarbonPeriod::create($currentFrom, '1 day', $currentTo);
        foreach ($period as $date) {
            $label = $date->format('d M');
            $labels[] = $label;

            $customerCount = Customer::whereDate('created_at', $date)->count();
            $leadCount = Lead::whereDate('created_at', $date)->count();

            $customerSeries[] = $customerCount;
            $leadSeries[] = $leadCount;
        }

        return response()->json([
            'total_premium' => $currentPremium,
            'premium_change' => $premiumChange,
            'total_leads' => $currentLeads,
            'leads_change' => $leadsChange,
            'total_policies_sold' => $currentPolicies,
            'policies_change' => $policiesChange,
            'chart_series' => [
                ['name' => 'Customers', 'type' => 'area', 'data' => $customerSeries],
                ['name' => 'Leads', 'type' => 'line', 'data' => $leadSeries],
            ],
            'labels' => $labels,
        ]);
    }

    private function getDateRange($range, $from = null, $to = null)
    {
        switch ($range) {
            case 'daily':
                return [Carbon::today(), Carbon::today()->endOfDay()];
            case 'weekly':
                return [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()];
            case 'monthly':
                return [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()];
            case 'yearly':
                return [Carbon::now()->startOfYear(), Carbon::now()->endOfYear()];
            case 'custom':
                if ($from && $to) {
                    return [Carbon::parse($from)->startOfDay(), Carbon::parse($to)->endOfDay()];
                }
                break;
        }
        return [Carbon::now()->startOfYear(), Carbon::now()->endOfYear()];
    }

    public function notificationStats()
    {
        $channels = ['sms', 'email', 'ivr'];
        $data = [];

        foreach ($channels as $channel) {
            $query = Notification::where('channel', $channel);
            $total = $query->count();
            $pending = (clone $query)->where('status', 'pending')->count();
            $sent = (clone $query)->where('status', 'sent')->count();
            $failed = (clone $query)->where('status', 'failed')->count();

            $data[$channel] = [
                'total' => $total,
                'pending' => $pending,
                'sent' => $sent,
                'failed' => $failed,
            ];
        }

        return response()->json($data);
    }

    // Return paginated filtered notifications by channel, status, search (AJAX for modal)
    public function channelData(Request $request)
    {
        $channel = $request->query('channel');
        $status = $request->query('status');
        $search = $request->query('search');
        $page = $request->query('page', 1);
        $perPage = 10;

        $query = Notification::where('channel', $channel);

        if ($status) {
            $query->where('status', $status);
        }

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%$search%")
                  ->orWhere('message', 'like', "%$search%");
            });
        }

        $notifications = $query->orderBy('scheduled_at', 'desc')
            ->paginate($perPage, ['*'], 'page', $page);

        // Build table rows HTML manually here (no Blade partials)
        $rowsHtml = '';
        foreach ($notifications as $notif) {
            $rowsHtml .= '<tr>';
            $rowsHtml .= '<td>' . e($notif->id) . '</td>';
            $rowsHtml .= '<td>' . e($notif->title) . '</td>';
            $rowsHtml .= '<td>' . e($notif->message) . '</td>';
            $rowsHtml .= '<td><span class="badge bg-' . $this->channelColor($notif->channel) . '">' . strtoupper(e($notif->channel)) . '</span></td>';
            $rowsHtml .= '<td><span class="badge bg-' . $this->statusColor($notif->status) . '">' . ucfirst(e($notif->status)) . '</span></td>';
            $rowsHtml .= '<td>' . e($notif->scheduled_at) . '</td>';
            $rowsHtml .= '<td>' . e($notif->created_at) . '</td>';
            $rowsHtml .= '</tr>';
        }

        // Build pagination HTML for Bootstrap 5 style
        $paginationHtml = $this->renderPagination($notifications);

        return response()->json([
            'tableRows' => $rowsHtml,
            'pagination' => $paginationHtml,
        ]);
    }

    // Optional: Export CSV
    public function export(Request $request)
    {
        $channel = $request->query('channel');
        $status = $request->query('status');
        $search = $request->query('search');

        $query = Notification::where('channel', $channel);

        if ($status) {
            $query->where('status', $status);
        }

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%$search%")
                  ->orWhere('message', 'like', "%$search%");
            });
        }

        $notifications = $query->orderBy('scheduled_at', 'desc')->get();

        $filename = "notifications_{$channel}_" . now()->format('Ymd_His') . ".csv";

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        $columns = ['ID', 'Title', 'Message', 'Channel', 'Status', 'Scheduled At', 'Created At'];

        $callback = function () use ($notifications, $columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            foreach ($notifications as $notification) {
                fputcsv($file, [
                    $notification->id,
                    $notification->title,
                    $notification->message,
                    strtoupper($notification->channel),
                    ucfirst($notification->status),
                    $notification->scheduled_at,
                    $notification->created_at,
                ]);
            }
            fclose($file);
        };

        return Response::stream($callback, 200, $headers);
    }

    // Helpers for badges colors
    private function channelColor($channel)
    {
        return match ($channel) {
            'sms' => 'success',
            'email' => 'primary',
            'ivr' => 'warning',
            default => 'secondary',
        };
    }

    private function statusColor($status)
    {
        return match ($status) {
            'pending' => 'secondary',
            'sent' => 'success',
            'failed' => 'danger',
            default => 'dark',
        };
    }

    // Render simple Bootstrap 5 pagination
    private function renderPagination($paginator)
    {
        if ($paginator->lastPage() <= 1) return '';

        $html = '<ul class="pagination justify-content-center">';

        // Previous page
        $prevDisabled = $paginator->onFirstPage() ? 'disabled' : '';
        $html .= '<li class="page-item ' . $prevDisabled . '">';
        $html .= '<a class="page-link" href="#" data-page="' . ($paginator->currentPage() - 1) . '">Previous</a>';
        $html .= '</li>';

        // Pages
        for ($i = 1; $i <= $paginator->lastPage(); $i++) {
            $active = $paginator->currentPage() === $i ? 'active' : '';
            $html .= '<li class="page-item ' . $active . '">';
            $html .= '<a class="page-link" href="#" data-page="' . $i . '">' . $i . '</a>';
            $html .= '</li>';
        }

        // Next page
        $nextDisabled = $paginator->currentPage() === $paginator->lastPage() ? 'disabled' : '';
        $html .= '<li class="page-item ' . $nextDisabled . '">';
        $html .= '<a class="page-link" href="#" data-page="' . ($paginator->currentPage() + 1) . '">Next</a>';
        $html .= '</li>';

        $html .= '</ul>';

        return $html;
    }

}
