<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Customer;
use App\Models\Policy;
use App\Models\Claim;
use App\Models\SurveyStatus;
use App\Models\GarageDischarge;
use App\Models\Notification;
use App\Models\NotificationPreference;
use Illuminate\Support\Carbon;

class NotificationSeeder extends Seeder
{
    public function run(): void
    {
        $customers = Customer::with(['policies', 'claims'])->get();

        foreach ($customers as $customer) {
            $preference = NotificationPreference::where('customer_id', $customer->id)->first();
            if (!$preference) {
                continue; // Skip customers without preferences
            }

            $language = $customer->preferred_language ?? 'en';

            // Policy Expiry Notifications
            foreach ($customer->policies as $policy) {
                if ($policy->expiry_date) {
                    $channels = $this->getAllowedChannels($preference);
                    foreach ($channels as $channel) {
                        Notification::create([
                            'customer_id' => $customer->id,
                            'title' => $language === 'am' ? 'የፖሊሲ ጊዜ መያዣ' : 'Policy Expiry Reminder',
                            'message' => $language === 'am'
                                ? "የእርስዎ ፖሊሲ {$policy->policy_number} በ " . Carbon::parse($policy->expiry_date)->format('Y-m-d') . " ይያዛል።"
                                : "Your policy {$policy->policy_number} will expire on " . Carbon::parse($policy->expiry_date)->format('Y-m-d') . ".",
                            'channel' => $channel,
                            'status' => 'pending',
                            'scheduled_at' => Carbon::now()->addDays(rand(1, 5)),
                        ]);
                    }
                }
            }

            // Claim Notifications
            foreach ($customer->claims as $claim) {
                $channels = $this->getAllowedChannels($preference);
                foreach ($channels as $channel) {
                    Notification::create([
                        'customer_id' => $customer->id,
                        'title' => $language === 'am' ? 'ክሊም ምዝገባ' : 'Claim Registered',
                        'message' => $language === 'am'
                            ? "የእርስዎ ክሊም {$claim->claim_number} ተመዝግቧል።"
                            : "Your claim {$claim->claim_number} has been registered.",
                        'channel' => $channel,
                        'status' => 'pending',
                        'scheduled_at' => Carbon::now()->addDays(rand(1, 5)),
                    ]);
                }
            }

            // Survey Status
            $surveys = SurveyStatus::where('customer_id', $customer->id)->get();
            foreach ($surveys as $survey) {
                if ($survey->survey_status === 'Completed') {
                    $channels = $this->getAllowedChannels($preference);
                    foreach ($channels as $channel) {
                        Notification::create([
                            'customer_id' => $customer->id,
                            'title' => $language === 'am' ? 'የእይታ ሪፖርት ተጠናቋል' : 'Survey Completed',
                            'message' => $language === 'am'
                                ? "የእርስዎ ፖሊሲ ላይ የተደረገው የእይታ ሪፖርት ተጠናቋል።"
                                : "The risk survey on your policy has been completed.",
                            'channel' => $channel,
                            'status' => 'pending',
                            'scheduled_at' => Carbon::now()->addDays(rand(1, 3)),
                        ]);
                    }
                }
            }

            // Garage Discharge
            $discharges = GarageDischarge::where('customer_id', $customer->id)->get();
            foreach ($discharges as $discharge) {
                $channels = $this->getAllowedChannels($preference);
                foreach ($channels as $channel) {
                    Notification::create([
                        'customer_id' => $customer->id,
                        'title' => $language === 'am' ? 'መኪና ከጋራጅ ተለቀቀ' : 'Vehicle Discharged from Garage',
                        'message' => $language === 'am'
                            ? "መኪናዎ ከ {$discharge->garage_name} በ " . Carbon::parse($discharge->discharge_date)->format('Y-m-d') . " ተለቀቀ።"
                            : "Your vehicle was discharged from {$discharge->garage_name} on " . Carbon::parse($discharge->discharge_date)->format('Y-m-d') . ".",
                        'channel' => $channel,
                        'status' => 'pending',
                        'scheduled_at' => Carbon::now()->addDays(rand(1, 4)),
                    ]);
                }
            }
        }
    }

    private function getAllowedChannels($preference): array
    {
        $channels = [];
        if ($preference->via_sms) $channels[] = 'sms';
        if ($preference->via_email) $channels[] = 'email';
        if ($preference->via_ivr) $channels[] = 'ivr';
        return $channels;
    }
}
