<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Claim;
use Carbon\Carbon;

class InvoiceController extends Controller
{
    /**
     * Return unpaid invoices summary for chart and total.
     */
    public function summary(Request $request)
    {
        $range = $request->input('range', 'all');
        $from = $request->input('from');
        $to = $request->input('to');

        $query = Claim::query()
            ->where('claim_status', 'pending');

        // Date filters
        if ($range === 'daily') {
            $query->whereDate('submitted_at', Carbon::today());
        } elseif ($range === 'weekly') {
            $query->whereBetween('submitted_at', [
                Carbon::now()->startOfWeek(),
                Carbon::now()->endOfWeek()
            ]);
        } elseif ($range === 'monthly') {
            $query->whereMonth('submitted_at', Carbon::now()->month)
                  ->whereYear('submitted_at', Carbon::now()->year);
        } elseif ($range === 'yearly') {
            $query->whereYear('submitted_at', Carbon::now()->year);
        } elseif ($range === 'custom' && $from && $to) {
            $query->whereBetween('submitted_at', [
                Carbon::parse($from)->startOfDay(),
                Carbon::parse($to)->endOfDay()
            ]);
        }

        $claims = $query->get();

        // Grouping for chart
        $chart = $claims->groupBy(function ($item) use ($range) {
            $date = Carbon::parse($item->submitted_at);

            return match ($range) {
                'daily' => $date->format('H:00'),
                'weekly', 'monthly', 'custom' => $date->format('d M'),
                'yearly' => $date->format('M'),
                default => $date->format('d M'),
            };
        })->map(function ($group, $label) {
            return [
                'label' => $label,
                'total' => $group->sum('claim_amount_requested'),
            ];
        })->values();

        return response()->json([
            'total_amount' => round($claims->sum('claim_amount_requested'), 2),
            'total_invoices' => $claims->count(),
            'chart' => $chart,
        ]);
    }

    private function getRangeDates($range)
    {
        switch ($range) {
            case 'daily':
                return [now()->startOfDay(), now()->endOfDay()];
            case 'weekly':
                return [now()->startOfWeek(), now()->endOfWeek()];
            case 'monthly':
                return [now()->startOfMonth(), now()->endOfMonth()];
            case 'yearly':
                return [now()->startOfYear(), now()->endOfYear()];
            default:
                return [now()->startOfMonth(), now()->endOfMonth()];
        }
    }

}
