<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Notification;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Carbon;

use Illuminate\Support\Facades\Mail;
use App\Mail\NotificationEmail;
use Illuminate\Support\Facades\Http;

use App\Services\SmsService;

use Google\Cloud\TextToSpeech\V1\Client\TextToSpeechClient;
use Google\Cloud\TextToSpeech\V1\SynthesizeSpeechRequest;
use Google\Cloud\TextToSpeech\V1\SynthesisInput;
use Google\Cloud\TextToSpeech\V1\VoiceSelectionParams;
use Google\Cloud\TextToSpeech\V1\AudioConfig;
use Google\Cloud\TextToSpeech\V1\AudioEncoding;

class NotificationController extends Controller
{
    public function index(Request $request)
    {
        if ($request->ajax()) {
            // Join customers to enable filtering by full_name
            $query = Notification::with('customer')
                ->leftJoin('customers', 'notifications.customer_id', '=', 'customers.id')
                ->select('notifications.*');

            // Filter by channel
            if ($request->filled('channel')) {
                $query->where('notifications.channel', $request->channel);
            }

            // Filter by status
            if ($request->filled('status')) {
                $query->where('notifications.status', $request->status);
            }

            // Filter by customer name
            if ($request->filled('customer_name')) {
                $query->where('customers.full_name', 'like', '%' . $request->customer_name . '%');
            }

            // Filter by title
            if ($request->filled('title')) {
                $query->where('notifications.title', 'like', '%' . $request->title . '%');
            }

            return DataTables::of($query)
            ->addColumn('customer', function ($notification) {
                $full_name = $notification->customer?->full_name ?? 'N/A';
                $initials = collect(explode(' ', $full_name))
                    ->map(fn($n) => strtoupper(substr($n, 0, 1)))
                    ->implode('');

                return '
                    <div class="d-flex align-items-center">
                        <span class="bg-primary text-white h-30 w-30 d-flex justify-content-center align-items-center rounded-circle fw-bold">'
                            . e($initials) . '
                        </span>
                        <p class="mb-0 ps-2">' . e($full_name) . '</p>
                    </div>';
            })
            ->editColumn('scheduled_at', function ($notification) {
                return $notification->scheduled_at
                    ? Carbon::parse($notification->scheduled_at)->format('F j, Y H:i:s')
                    : '';
            })
            ->editColumn('created_at', function ($notification) {
                return $notification->created_at
                    ? Carbon::parse($notification->created_at)->format('F j, Y H:i:s')
                    : '';
            })
            ->addColumn('actions', function ($notification) {
                return '
                    <button class="btn btn-light-success icon-btn b-r-4 me-1 openNotificationModal"
                        data-id="' . $notification->id . '"
                        title="View Details">
                        <i class="ti ti-eye"></i>
                    </button>

                    <button id="" class="btn btn-light-primary icon-btn b-r-4 sendNotificationBtn"
                        data-id="' . $notification->id . '"
                        title="Send Notification"
                        data-bs-toggle="modal"
                        data-bs-target="#sendNotifModal">
                        <i class="ti ti-send"></i>
                    </button>
                ';
            })
            ->rawColumns(['customer', 'actions'])
            ->make(true);
        }

        return view('admin.notifications.index');
    }

    public function show($id)
    {
        $notification = Notification::with([
            'customer',
            'claims',
            'policies',
            'claimPayments',
            'garageDischarges',
            'surveyStatuses',
            'workOrders',
        ])->findOrFail($id);

        return view('admin.notifications.partials.modal-tabs', compact('notification'));
    }





public function send(Request $request)
{
    $channels = $request->input('channels', []);
    $phone = $request->input('phone_number');
    $email = $request->input('email');
    $smsMessage = $request->input('sms_message');
    $emailMessage = $request->input('email_message');
    $ivrScript = $request->input('ivr_script');

    $responses = [];

    try {
        // Send via SMS (GeezSMS API)
        if (in_array('sms', $channels) && $phone) {
            $geezToken = '0mcUvCaKk7S6lIB7ZXoBsMaQ32LDQIKA';
            $geezUrl = 'https://api.geezsms.com/api/v1/sms/send';

            $smsResponse = Http::get($geezUrl, [
                'token' => $geezToken,
                'phone' => '0911525523',
                'msg'   => $smsMessage,
            ]);

            $responses['sms'] = $smsResponse->json();
        }

        // Send via Email
        if (in_array('email', $channels) && $email) {
            Mail::raw($emailMessage, function ($message) use ($email) {
                $message->to($email)
                        ->subject('Notification Message');
            });

            $responses['email'] = 'Email sent successfully';
        }

        // IVR - placeholder for integration
        if (in_array('ivr', $channels)) {
            $responses['ivr'] = 'IVR sending simulated (no API configured yet)';
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Notification(s) sent successfully!',
            'responses' => $responses,
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'status' => 'error',
            'message' => 'Failed to send notification.',
            'error' => $e->getMessage(),
        ], 500);
    }
}



    public function show_id($id)
    {
        $notification = Notification::with('customer')->findOrFail($id);

        return response()->json([
            'id' => $notification->id,
            'message' => $notification->message,
            'customer' => [
                'full_name' => $notification->customer->full_name ?? '',
                'phone_number' => $notification->customer->phone_number ?? '',
                'email' => $notification->customer->email ?? '',
            ]
        ]);
    }

public function generateIVRAudio(Request $request)
{
    $request->validate([
        'text' => 'required|string|max:1000',
    ]);

    try {
        $client = new TextToSpeechClient();

        $input = new SynthesisInput();
        $input->setText($request->text);

        $voice = new VoiceSelectionParams([
            'language_code' => 'en-US',
            'ssml_gender' => 'NEUTRAL',
        ]);

        $audioConfig = new AudioConfig([
            'audio_encoding' => AudioEncoding::MP3,
        ]);

        $response = $client->synthesizeSpeech($input, $voice, $audioConfig);
        $audioContent = $response->getAudioContent();

        $filename = 'ivr_' . now()->timestamp . '.mp3';
        $path = storage_path('app/public/' . $filename);
        file_put_contents($path, $audioContent);

        $client->close();

        return response()->json([
            'status' => 'success',
            'audio_url' => asset('storage/' . $filename),
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'status' => 'error',
            'message' => 'Failed to generate audio: ' . $e->getMessage(),
        ], 500);
    }
}


}
