<?php

namespace App\Http\Controllers\Admin;


use App\Http\Controllers\Controller;
use App\Models\Policy;
use Carbon\Carbon;

use App\Models\Claim;
use Illuminate\Http\Request;

class PolicyController extends Controller
{
    //

public function getByCustomer($customerId)
{
    $policies = Policy::where('customer_id', $customerId)
        ->select('id', 'policy_number')
        ->orderBy('policy_number')
        ->get();

    $results = $policies->map(function ($policy) {
        return [
            'id' => $policy->id,
            'text' => $policy->policy_number,
        ];
    });

    return response()->json(['results' => $results]);
}

public function details($id)
{
    $policy = Policy::find($id);

    if (!$policy) {
        return response()->json(['error' => 'Policy not found'], 404);
    }

    return response()->json([
        'status' => ucfirst($policy->status),
        'policy_name' => $policy->policy_name,
        'description' => $policy->description,
        'start_date' => $policy->start_date ? Carbon::parse($policy->start_date)->format('Y-m-d') : null,
        'end_date' => $policy->end_date ? Carbon::parse($policy->end_date)->format('Y-m-d') : null,
        'premium_amount' => $policy->premium_amount,
    ]);
}

public function show($id)
{
    $policy = \App\Models\Policy::find($id);

    if (!$policy) {
        return response()->json(['message' => 'Policy not found'], 404);
    }

    return response()->json([
        'id' => $policy->id,
        'policy_number' => $policy->policy_number,
        'policy_type' => $policy->policy_type,
        'coverage_type' => $policy->coverage_type,
        'start_date' => $policy->start_date,
        'end_date' => $policy->end_date,
        'premium_amount' => $policy->premium_amount,
        'status' => $policy->status,
        // add any other fields needed for the view
    ]);
}
public function loadTab($id, $tab)
{
    $policy = Policy::with('customer', 'claims')->findOrFail($id);

    switch ($tab) {
        case 'customer':
            $customer = $policy->customer;
            $html = "
                <div>
                    <h6>Name:</h6> {$customer->first_name} {$customer->last_name}<br>
                    <strong>Email:</strong> {$customer->email}<br>
                    <strong>Phone:</strong> {$customer->phone}
                </div>";
            break;

        case 'policy':
            $html = "
                <div>
                    <h6>Policy Number:</h6> {$policy->policy_number}<br>
                    <strong>Type:</strong> {$policy->policy_type}<br>
                    <strong>Premium:</strong> {$policy->premium_amount} ETB<br>
                    <strong>Status:</strong> <span class='badge bg-success'>{$policy->status}</span>
                </div>";
            break;

        case 'claims':
            if ($policy->claims->isEmpty()) {
                $html = "<div class='text-muted'>No claims found for this policy.</div>";
            } else {
                $html = "<ul class='list-group'>";
                foreach ($policy->claims as $claim) {
                    $html .= "<li class='list-group-item'>
                        <strong>Claim #{$claim->id}</strong><br>
                        Amount: <strong>{$claim->claim_amount} ETB</strong><br>
                        Status: <span class='badge bg-info'>{$claim->status}</span><br>
                        Filed on: {$claim->created_at->format('d M Y')}
                    </li>";
                }
                $html .= "</ul>";
            }
            break;

        default:
            return response()->json(['data' => "<div class='text-danger'>Invalid tab selected.</div>"], 400);
    }

    return response()->json(['data' => $html]);
}


// Controller: PolicyController.php

public function ajaxCustomer($id)
{
    $policy = \App\Models\Policy::with('customer')->findOrFail($id);
    return response()->json([
        'html' => view('admin.policies.inline.customer', compact('policy'))->render()
    ]);
}

public function ajaxPolicy($id)
{
    $policy = \App\Models\Policy::with('beneficiaries')->findOrFail($id);
    return response()->json([
        'html' => view('admin.policies.inline.policy', compact('policy'))->render()
    ]);
}

public function ajaxClaims($id)
{
    $claims = \App\Models\Claim::where('policy_id', $id)->latest()->get();
    return response()->json([
        'html' => view('admin.policies.inline.claims', compact('claims'))->render()
    ]);
}

}
