<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Validation\ValidationException;
use App\Models\LoginAttempt;

class LoginController extends Controller
{
    use AuthenticatesUsers;

    public function authenticated()
    {
        if (request()->wantsJson()) {
            return response()->json(['success' => true]);
        }

        return redirect()->route('admin.dashboard');
    }

    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            $this->logLoginAttempt($request, 'failed', 'Validation failed');
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $throttleKey = strtolower($request->input('email')) . '|' . $request->ip();

        if (RateLimiter::tooManyAttempts($throttleKey, 5)) {
            $seconds = RateLimiter::availableIn($throttleKey);
            $this->logLoginAttempt($request, 'failed', "Throttled: retry after {$seconds} seconds");

            return response()->json([
                'error' => 'Too many login attempts. Please try again later.',
                'retry_after' => $seconds,
            ], 429);
        }

        $credentials = $request->only('email', 'password');
        $user = \App\Models\User::where('email', $credentials['email'])->first();

        if ($user) {
            if ($user->is_locked) {
                $this->logLoginAttempt($request, 'failed', 'Account locked');
                return response()->json(['error' => 'Your account is locked due to multiple failed login attempts. Please contact support.'], 423);
            }

            if ($user->status != 1) {
                $this->logLoginAttempt($request, 'failed', 'Account inactive');
                return response()->json(['error' => 'Your account is inactive. Please contact support.'], 403);
            }

            if (is_null($user->email_verified_at)) {
                $this->logLoginAttempt($request, 'failed', 'Email not verified');
                return response()->json(['error' => 'Please verify your email before logging in.'], 403);
            }

            if (Auth::attempt($credentials, $request->has('remember'))) {
                $user->update(['login_attempts' => 0]);
                RateLimiter::clear($throttleKey);

                $this->logLoginAttempt($request, 'successful', 'Login successful');

                return response()->json(['success' => true]);
            }
        }

        // Failed login (user not found or password mismatch)
        if ($user) {
            $user->increment('login_attempts');
            if ($user->login_attempts >= 5) {
                $user->update(['is_locked' => true]);
            }
        }

        RateLimiter::hit($throttleKey, 60);
        $this->logLoginAttempt($request, 'failed', 'Invalid credentials');

        return response()->json(['error' => 'Incorrect username or password. Please try again.'], 401);
    }

    /**
     * Log login attempts.
     */
    private function logLoginAttempt(Request $request, $status, $reason)
    {
        $user = \App\Models\User::where('email', $request->input('email'))->first();

        LoginAttempt::create([
            'user_id' => optional($user)->id,
            'email' => $request->input('email'),
            'ip_address' => $request->ip(),
            'status' => $status,
            'reason' => $reason,
        ]);
    }

    public function __construct()
    {
        $this->middleware('guest')->except('logout');
        $this->middleware('auth')->only('logout');
    }
}
