<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Notification;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Carbon;

use App\Services\SmsService;

class NotificationController extends Controller
{
    public function index(Request $request)
    {
        if ($request->ajax()) {
            // Join customers to enable filtering by full_name
            $query = Notification::with('customer')
                ->leftJoin('customers', 'notifications.customer_id', '=', 'customers.id')
                ->select('notifications.*');

            // Filter by channel
            if ($request->filled('channel')) {
                $query->where('notifications.channel', $request->channel);
            }

            // Filter by status
            if ($request->filled('status')) {
                $query->where('notifications.status', $request->status);
            }

            // Filter by customer name
            if ($request->filled('customer_name')) {
                $query->where('customers.full_name', 'like', '%' . $request->customer_name . '%');
            }

            // Filter by title
            if ($request->filled('title')) {
                $query->where('notifications.title', 'like', '%' . $request->title . '%');
            }

            return DataTables::of($query)
            ->addColumn('customer', function ($notification) {
                $full_name = $notification->customer?->full_name ?? 'N/A';
                $initials = collect(explode(' ', $full_name))
                    ->map(fn($n) => strtoupper(substr($n, 0, 1)))
                    ->implode('');

                return '
                    <div class="d-flex align-items-center">
                        <span class="bg-primary text-white h-30 w-30 d-flex justify-content-center align-items-center rounded-circle fw-bold">'
                            . e($initials) . '
                        </span>
                        <p class="mb-0 ps-2">' . e($full_name) . '</p>
                    </div>';
            })
            ->editColumn('scheduled_at', function ($notification) {
                return $notification->scheduled_at
                    ? Carbon::parse($notification->scheduled_at)->format('F j, Y H:i:s')
                    : '';
            })
            ->editColumn('created_at', function ($notification) {
                return $notification->created_at
                    ? Carbon::parse($notification->created_at)->format('F j, Y H:i:s')
                    : '';
            })
            ->addColumn('actions', function ($notification) {
                return '
                    <button class="btn btn-light-success icon-btn b-r-4 me-1 openNotificationModal"
                        data-id="' . $notification->id . '"
                        title="View Details">
                        <i class="ti ti-eye"></i>
                    </button>

                    <button id="sendNotificationBtn" class="btn btn-light-primary icon-btn b-r-4 "
                        data-id="' . $notification->id . '"
                        title="Send Notification"
                        data-bs-toggle="modal"
                        data-bs-target="#sendNotifModal">
                        <i class="ti ti-send"></i>
                    </button>
                ';
            })
            ->rawColumns(['customer', 'actions'])
            ->make(true);
        }

        return view('admin.notifications.index');
    }

    public function show($id)
    {
        $notification = Notification::with([
            'customer',
            'claims',
            'policies',
            'claimPayments',
            'garageDischarges',
            'surveyStatuses',
            'workOrders',
        ])->findOrFail($id);

        return view('admin.notifications.partials.modal-tabs', compact('notification'));
    }



    public function send(Request $request, SmsService $smsService)
    {
     // Sample data
     $to = '+251911525523';
     $policyNumber = 'POL123456789';
     $message = "Dear customer, your policy number {$policyNumber} is active and valid.";

     // Call the SMS service to send the message
     $result = $smsService->sendSms($to, $message);

     if ($result['status'] === 'success') {
         return response()->json([
             'message' => 'SMS sent successfully.',
             'to' => $to,
             'text' => $message,
         ]);
     }

     return response()->json([
         'message' => 'Failed to send SMS.',
         'error' => $result['error'] ?? 'Unknown error',
     ], 500);
    }


    public function show_id($id)
    {
        $notification = Notification::with('customer')->findOrFail($id);

        return response()->json([
            'id' => $notification->id,
            'message' => $notification->message,
            'customer' => [
                'full_name' => $notification->customer->full_name ?? '',
                'phone_number' => $notification->customer->phone_number ?? '',
                'email' => $notification->customer->email ?? '',
            ]
        ]);
    }




}
